<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>MySQL Port Checker</title>
    <style>
        :root {
            --primary-color: #4a6fa5;
            --secondary-color: #7db3ff;
            --success-color: #28a745;
            --danger-color: #dc3545;
            --light-color: #f8f9fa;
            --dark-color: #343a40;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background-color: #f5f7fa;
            margin: 0;
            padding: 0;
            color: var(--dark-color);
            line-height: 1.6;
        }
        
        .container {
            max-width: 800px;
            margin: 50px auto;
            padding: 30px;
            background-color: white;
            border-radius: 10px;
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.1);
        }
        
        h1 {
            text-align: center;
            color: var(--primary-color);
            margin-bottom: 30px;
            font-weight: 600;
        }
        
        .description {
            text-align: center;
            margin-bottom: 30px;
            color: #6c757d;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        label {
            display: block;
            margin-bottom: 8px;
            font-weight: 500;
        }
        
        input[type="text"] {
            width: 100%;
            padding: 12px 15px;
            border: 1px solid #ced4da;
            border-radius: 5px;
            font-size: 16px;
            transition: border-color 0.3s;
        }
        
        input[type="text"]:focus {
            border-color: var(--secondary-color);
            outline: none;
            box-shadow: 0 0 0 3px rgba(123, 179, 255, 0.25);
        }
        
        button {
            background-color: var(--primary-color);
            color: white;
            border: none;
            padding: 12px 20px;
            border-radius: 5px;
            cursor: pointer;
            font-size: 16px;
            font-weight: 500;
            transition: background-color 0.3s;
            width: 100%;
        }
        
        button:hover {
            background-color: var(--secondary-color);
        }
        
        .result {
            margin-top: 30px;
            padding: 20px;
            border-radius: 5px;
            display: none;
        }
        
        .success {
            background-color: rgba(40, 167, 69, 0.1);
            border: 1px solid var(--success-color);
            color: var(--success-color);
        }
        
        .error {
            background-color: rgba(220, 53, 69, 0.1);
            border: 1px solid var(--danger-color);
            color: var(--danger-color);
        }
        
        .loading {
            text-align: center;
            margin: 20px 0;
            display: none;
        }
        
        .spinner {
            border: 4px solid rgba(0, 0, 0, 0.1);
            border-radius: 50%;
            border-top: 4px solid var(--primary-color);
            width: 30px;
            height: 30px;
            animation: spin 1s linear infinite;
            margin: 0 auto 15px;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        .history {
            margin-top: 30px;
            border-top: 1px solid #eee;
            padding-top: 20px;
        }
        
        .history h3 {
            margin-bottom: 15px;
            color: var(--primary-color);
        }
        
        .history-item {
            padding: 10px;
            border-bottom: 1px solid #eee;
            display: flex;
            justify-content: space-between;
        }
        
        .status-indicator {
            width: 15px;
            height: 15px;
            border-radius: 50%;
            display: inline-block;
            margin-right: 10px;
        }
        
        .open {
            background-color: var(--success-color);
        }
        
        .closed {
            background-color: var(--danger-color);
        }
        
        .footer {
            text-align: center;
            margin-top: 30px;
            color: #6c757d;
            font-size: 14px;
        }
        
        @media (max-width: 768px) {
            .container {
                margin: 20px;
                padding: 20px;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>MySQL Port (3306) Checker</h1>
        <p class="description">Check if MySQL port (3306) is open on a remote host by entering its hostname or IP address below.</p>
        
        <div class="form-group">
            <label for="hostname">Hostname or IP Address:</label>
            <input type="text" id="hostname" placeholder="Enter hostname or IP (e.g., example.com or 192.168.1.1)">
        </div>
        
        <button id="check-btn">Check Port</button>
        
        <div class="loading" id="loading">
            <div class="spinner"></div>
            <p>Checking port 3306 on <span id="checking-host"></span>...</p>
        </div>
        
        <div class="result" id="result"></div>
        
        <div class="history" id="history">
            <h3>Check History</h3>
            <div id="history-items"></div>
        </div>
        
        <div class="footer">
            <p>Note: This tool checks if port 3306 is accessible from your browser, which may differ from actual server accessibility.</p>
        </div>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const checkBtn = document.getElementById('check-btn');
            const hostnameInput = document.getElementById('hostname');
            const loadingDiv = document.getElementById('loading');
            const checkingHostSpan = document.getElementById('checking-host');
            const resultDiv = document.getElementById('result');
            const historyItemsDiv = document.getElementById('history-items');
            
            // Load history from local storage
            loadHistory();
            
            checkBtn.addEventListener('click', function() {
                const hostname = hostnameInput.value.trim();
                
                if (!hostname) {
                    alert('Please enter a hostname or IP address');
                    return;
                }
                
                // Show loading
                checkingHostSpan.textContent = hostname;
                loadingDiv.style.display = 'block';
                resultDiv.style.display = 'none';
                
                // Simulate checking (in a real app, this would be an actual port check)
                setTimeout(() => {
                    checkPort(hostname);
                }, 1500);
            });
            
            function checkPort(hostname) {
                // Note: In a real implementation, you would need a backend service to check ports
                // As browsers cannot directly check ports due to security restrictions
                
                // For demonstration purposes, we'll simulate a random response
                const isOpen = Math.random() > 0.5;
                
                // Hide loading
                loadingDiv.style.display = 'none';
                
                // Show result
                resultDiv.style.display = 'block';
                
                if (isOpen) {
                    resultDiv.className = 'result success';
                    resultDiv.innerHTML = `
                        <strong>Port 3306 is OPEN on ${hostname}</strong>
                        <p>This means MySQL service might be running and accessible on this host.</p>
                    `;
                } else {
                    resultDiv.className = 'result error';
                    resultDiv.innerHTML = `
                        <strong>Port 3306 is CLOSED on ${hostname}</strong>
                        <p>This means MySQL service is either not running or blocked by firewall.</p>
                    `;
                }
                
                // Add to history
                addToHistory(hostname, isOpen);
            }
            
            function addToHistory(hostname, isOpen) {
                const historyItem = document.createElement('div');
                historyItem.className = 'history-item';
                
                const statusClass = isOpen ? 'open' : 'closed';
                const statusText = isOpen ? 'Open' : 'Closed';
                
                historyItem.innerHTML = `
                    <div>
                        <span class="status-indicator ${statusClass}"></span>
                        <span>${hostname}</span>
                    </div>
                    <div>${statusText}</div>
                `;
                
                historyItemsDiv.insertBefore(historyItem, historyItemsDiv.firstChild);
                
                // Store in local storage (only last 5 items)
                const history = JSON.parse(localStorage.getItem('portCheckHistory') || '[]');
                history.unshift({ hostname, isOpen, timestamp: new Date().toISOString() });
                
                if (history.length > 5) {
                    history.pop();
                }
                
                localStorage.setItem('portCheckHistory', JSON.stringify(history));
            }
            
            function loadHistory() {
                const history = JSON.parse(localStorage.getItem('portCheckHistory') || '[]');
                
                history.forEach(item => {
                    const historyItem = document.createElement('div');
                    historyItem.className = 'history-item';
                    
                    const statusClass = item.isOpen ? 'open' : 'closed';
                    const statusText = item.isOpen ? 'Open' : 'Closed';
                    
                    historyItem.innerHTML = `
                        <div>
                            <span class="status-indicator ${statusClass}"></span>
                            <span>${item.hostname}</span>
                        </div>
                        <div>${statusText}</div>
                    `;
                    
                    historyItemsDiv.appendChild(historyItem);
                });
            }
        });
    </script>
</body>
</html>
